"use client";

import { Role } from "@prisma/client";
import { useMemo, useState } from "react";

type UserRecord = {
  id: string;
  firstName: string;
  lastName: string;
  email: string;
  role: Role;
  createdAt: Date;
};

type LogRecord = {
  id: string;
  action: string;
  target: string | null;
  description: string | null;
  createdAt: Date;
  actor: { firstName: string; lastName: string; email: string } | null;
};

export function AdminPanel({ users, logs }: { users: UserRecord[]; logs: LogRecord[] }) {
  const [search, setSearch] = useState("");
  const [stateUsers, setStateUsers] = useState(users);

  const filteredUsers = useMemo(() => {
    if (!search.trim()) return stateUsers;
    const q = search.toLowerCase();
    return stateUsers.filter((user) =>
      [user.firstName, user.lastName, user.email, user.role].join(" ").toLowerCase().includes(q)
    );
  }, [search, stateUsers]);

  async function updateRole(userId: string, role: Role) {
    const res = await fetch(`/api/admin/users/${userId}/role`, {
      method: "PATCH",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ role })
    });

    if (res.ok) {
      setStateUsers((prev) => prev.map((u) => (u.id === userId ? { ...u, role } : u)));
    }
  }

  async function triggerReset(userId: string) {
    const res = await fetch(`/api/admin/users/${userId}/reset-password`, { method: "POST" });
    const data = await res.json();
    if (res.ok) {
      alert(`Sıfırlama linki üretildi: ${data.resetLink}`);
    }
  }

  return (
    <div className="space-y-10">
      <section className="space-y-4 rounded-xl border border-slate-800 bg-slate-900/50 p-6">
        <div className="flex items-center justify-between gap-3">
          <h2 className="text-xl font-semibold">Kullanıcı Yönetimi</h2>
          <input
            value={search}
            onChange={(e) => setSearch(e.target.value)}
            placeholder="Ad, soyad, e-posta veya rol ara"
            className="w-full max-w-xs rounded bg-slate-800 p-2 text-sm"
          />
        </div>
        <div className="overflow-x-auto">
          <table className="min-w-full text-sm">
            <thead>
              <tr className="text-left text-slate-400">
                <th className="px-2 py-2">Ad Soyad</th>
                <th className="px-2 py-2">E-posta</th>
                <th className="px-2 py-2">Rol</th>
                <th className="px-2 py-2">İşlem</th>
              </tr>
            </thead>
            <tbody>
              {filteredUsers.map((user) => (
                <tr key={user.id} className="border-t border-slate-800">
                  <td className="px-2 py-2">
                    {user.firstName} {user.lastName}
                  </td>
                  <td className="px-2 py-2">{user.email}</td>
                  <td className="px-2 py-2">
                    <select
                      className="rounded bg-slate-800 p-1"
                      value={user.role}
                      onChange={(e) => updateRole(user.id, e.target.value as Role)}
                    >
                      <option value="USER">USER</option>
                      <option value="MODERATOR">MODERATOR</option>
                      <option value="ADMIN">ADMIN</option>
                    </select>
                  </td>
                  <td className="px-2 py-2">
                    <button className="rounded bg-amber-600 px-2 py-1" onClick={() => triggerReset(user.id)}>
                      Şifre Sıfırlama Linki
                    </button>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </section>

      <section className="space-y-4 rounded-xl border border-slate-800 bg-slate-900/50 p-6">
        <h2 className="text-xl font-semibold">Sistem Logları</h2>
        <ul className="space-y-2 text-sm">
          {logs.map((log) => (
            <li key={log.id} className="rounded border border-slate-800 p-3">
              <p>
                <strong>{log.action}</strong> → {log.target ?? "-"}
              </p>
              <p className="text-slate-300">{log.description ?? "Açıklama yok"}</p>
              <p className="text-xs text-slate-500">
                Actor: {log.actor ? `${log.actor.firstName} ${log.actor.lastName} (${log.actor.email})` : "System"} |{" "}
                {new Date(log.createdAt).toLocaleString("tr-TR")}
              </p>
            </li>
          ))}
        </ul>
      </section>
    </div>
  );
}
