"use client";

import { useState } from "react";
import { useRouter } from "next/navigation";

type Mode = "login" | "register";

export function AuthForm({ mode }: { mode: Mode }) {
  const router = useRouter();
  const [error, setError] = useState<string | null>(null);
  const [loading, setLoading] = useState(false);

  async function onSubmit(formData: FormData) {
    setError(null);
    setLoading(true);

    const endpoint = mode === "login" ? "/api/auth/login" : "/api/auth/register";
    const payload = Object.fromEntries(formData.entries());

    const response = await fetch(endpoint, {
      method: "POST",
      body: JSON.stringify(payload),
      headers: { "Content-Type": "application/json" }
    });

    setLoading(false);

    if (!response.ok) {
      const data = await response.json();
      setError(data.error ?? "İşlem başarısız.");
      return;
    }

    router.push(mode === "login" ? "/" : "/auth/login");
    router.refresh();
  }

  return (
    <form
      action={onSubmit}
      className="mx-auto flex max-w-md flex-col gap-3 rounded-xl border border-slate-800 bg-slate-900/50 p-6"
    >
      {mode === "register" && (
        <>
          <input name="firstName" placeholder="Ad" className="rounded bg-slate-800 p-2" required />
          <input name="lastName" placeholder="Soyad" className="rounded bg-slate-800 p-2" required />
        </>
      )}
      <input name="email" type="email" placeholder="E-posta" className="rounded bg-slate-800 p-2" required />
      <input name="password" type="password" placeholder="Şifre" className="rounded bg-slate-800 p-2" required />
      {error && <p className="text-sm text-rose-400">{error}</p>}
      <button type="submit" disabled={loading} className="rounded bg-cyan-600 px-3 py-2 font-medium text-white">
        {loading ? "Yükleniyor..." : mode === "login" ? "Giriş Yap" : "Kayıt Ol"}
      </button>
    </form>
  );
}
